<?php

namespace App\Console\Commands;

use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;
use App\Models\SeatSharingRequest;
use App\Models\SeatSharingVehicleLayout;
use App\Http\Controllers\API\v1\GcmController;

class AutoCancelRequest extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'app:auto-cancel-request-command';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Automatically cancel seat sharing rides that are older than 1 hour and not started';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        Log::info('AutoCancelRequest Cron Started');

        $cutoffTime = Carbon::now()->subHour(); // Rides older than 1 hour

        // Fetch all rides that are pending and departure_time is more than 1 hour ago
        $ridesToCancel = SeatSharingRequest::where('ride_status', 0)
            ->whereRaw("STR_TO_DATE(departure_time, '%Y-%m-%dT%H:%i:%s') <= ?", [$cutoffTime])
            ->get();

        Log::info("Found " . count($ridesToCancel) . " rides to auto-cancel.");

        foreach ($ridesToCancel as $ride) {
            // Update ride status to cancelled
            $ride->ride_status = 2;
            $ride->save();

            Log::info("Ride ID {$ride->id} auto-cancelled.");

            // Get all passenger user IDs
            $passengerUserIds = SeatSharingVehicleLayout::where('seat_sharing_request_id', $ride->id)
                ->whereNotNull('booked_by')
                ->pluck('booked_by')
                ->unique()
                ->toArray();

        Log::info("User IDs: " . json_encode($passengerUserIds));
            if (!empty($passengerUserIds)) {
                // Get FCM tokens of passengers
                $passengerFcmTokens = DB::table('tj_user_app')
                    ->whereIn('id', $passengerUserIds)
                    ->where('fcm_id', '!=', '')
                    ->pluck('fcm_id')
                    ->toArray();

                $title = "Ride Cancelled";
                $notif_msg = "Your ride from {$ride->from} to {$ride->to} has been cancelled due to no activity.";

                $message = [
                    "body" => $notif_msg,
                    "title" => $title,
                    "sound" => "default",
                    "tag" => "ride_cancel"
                ];

                // Send FCM notification
                foreach ($passengerFcmTokens as $token) {
                    GcmController::sendNotification($token, $message);
                }

                // Insert into notification table
                $notifications = [];
                $now = now();
                foreach ($passengerUserIds as $userId) {
                    $notifications[] = [
                        'titre' => $title,
                        'message' => $notif_msg,
                        'statut' => 'yes',
                        'creer' => $now,
                        'modifier' => $now,
                        'to_id' => $userId,
                        'from_id' => $ride->driver_id,
                        'type' => 'ride_cancel'
                    ];
                }

                DB::table('tj_notification')->insert($notifications);

                Log::info("Notifications sent to passengers for ride ID {$ride->id}");
            } else {
                Log::info("No passengers to notify for ride ID {$ride->id}");
            }
        }

        Log::info('AutoCancelRequest Cron Ended');
    }
}
